using System;
using System.Data;
using System.Reflection;

namespace gov.va.med.vbecs.BOL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary></summary>

		#endregion

	public class Reagent : InventoryItem
	{
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1259"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid reagent object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="673"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty constructor
		/// </summary>
		public Reagent()
		{
			_reagentGuid = Guid.Empty;
			_reagentName = string.Empty;
			_receiptDate = DateTime.MinValue;
			_lotNumber = string.Empty;
			_lotExpirationDate = DateTime.MinValue;
			_numberVialsReceived = 0;
			_numberVialsAvailable = 0;

			_reagentType = new ReagentType();
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2418"> 
		///		<ExpectedInput>DataRow containing Reagent data</ExpectedInput>
		///		<ExpectedOutput>Reagent object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="2861"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>NullReferenceException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates Reagent object and populates it with data from DataRow
		/// </summary>
		public Reagent(DataRow dr)
		{
			_reagentType = new ReagentType();
			this.LoadFromDataRow(dr);
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6626"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6627"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="reagentGuid"></param>
		public Reagent(Guid reagentGuid)
		{
			DataRow dr = BOL.Reagent.GetReagent(reagentGuid);
			_reagentType = new ReagentType();
			this.LoadFromDataRow(dr);
		}

		/// <summary>
		/// The list of columns
		/// </summary>
		public const string REAGENTCOLUMNS = "ReagentGuid,ReagentName,LotNumber,LotExpirationDate,"+
			"ReagentManufacturerName,InspectionIndicator";

		private Guid _reagentGuid;
		private string _reagentName;
		private DateTime _receiptDate;
		private string _lotNumber;
		private DateTime _lotExpirationDate;
		private short _numberVialsReceived;
		private short _numberVialsAvailable;
//		private short _testPerVial;
		private ReagentType _reagentType;
		private string _reagentManufacturerName;
		private bool _inspectionIndicator;

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1036"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>Valid GUID</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="746"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Reagent GUID
		/// </summary>
		public Guid ReagentGuid
		{
			get
			{
				return _reagentGuid;
			}
			set
			{
				_reagentGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1073"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="756"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReagentName
		/// </summary>
		public string ReagentName
		{
			get
			{
				return _reagentName;
			}
			set
			{
				_reagentName = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1074"> 
		///		<ExpectedInput>Valid DateTime</ExpectedInput>
		///		<ExpectedOutput>Valid DateTime</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="760"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Receipt date 
		/// </summary>
		public DateTime ReceiptDate
		{
			get
			{
				return _receiptDate;
			}
			set
			{
				_receiptDate = value;
			}
		}


		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1037"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="727"> 
		///		<ExpectedInput>empty string</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Lot number 
		/// </summary>
		public string LotNumber
		{
			get
			{
				return _lotNumber;
			}
			set
			{
				_lotNumber = value;
				//
				RuleBroken(MethodBase.GetCurrentMethod().Name,((_lotNumber==null)||(_lotNumber.Length==0)));
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1038"> 
		///		<ExpectedInput>Valid date and time</ExpectedInput>
		///		<ExpectedOutput>Valid date and time</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="719"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Lot Expiration Date
		/// </summary>
		public DateTime LotExpirationDate
		{
			get
			{
				return _lotExpirationDate;
			}
			set
			{
				_lotExpirationDate = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1039"> 
		///		<ExpectedInput>Valid short</ExpectedInput>
		///		<ExpectedOutput>Valid short</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="732"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Number of vials received
		/// </summary>
		public short NumberVialsReceived
		{
			get
			{
				return _numberVialsReceived;
			}
			set
			{
				_numberVialsReceived = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1040"> 
		///		<ExpectedInput>Valid short</ExpectedInput>
		///		<ExpectedOutput>Valid short</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="728"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Number of vials available
		/// </summary>
		public short NumberVialsAvailable
		{
			get
			{
				return _numberVialsAvailable;
			}
			set
			{
				_numberVialsAvailable = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1042"> 
		///		<ExpectedInput>Valid ReagentType</ExpectedInput>
		///		<ExpectedOutput>Valid ReagentType</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="759"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Reagent Type
		/// </summary>
		public ReagentType ReagentType
		{
			get
			{
				return _reagentType;
			}
			set
			{
				_reagentType = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2012"> 
		///		<ExpectedInput>String</ExpectedInput>
		///		<ExpectedOutput>String</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="749"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Reagent Manufacturer Name
		/// </summary>
		public string ReagentManufacturerName
		{
			get
			{
				return _reagentManufacturerName;
			}
			set
			{
				_reagentManufacturerName = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2013"> 
		///		<ExpectedInput>Boolean value</ExpectedInput>
		///		<ExpectedOutput>Boolean value</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="693"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inspection indicator
		/// </summary>
		public bool InspectionIndicator
		{
			get
			{
				return _inspectionIndicator;
			}
			set
			{
				_inspectionIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1075"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2002"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  returns a DataTable of reagents
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetReagents(string divisionCode)
		{
			return DAL.Reagent.GetReagents(divisionCode, DateTime.MaxValue, DateTime.MinValue, DateTime.MaxValue, DateTime.MinValue);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6651"> 
		///		<ExpectedInput>Valid DivisionCode, Valid dates</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6652"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return a DataTable of reagents
		/// </summary>
		/// <param name="_sDivisionCode"></param>
		/// <param name="ExpirationDateBefore"></param>
		/// <param name="ExpirationDateAfter"></param>
		/// <param name="ReceivedDateBefore"></param>
		/// <param name="ReceivedDateAfter"></param>
		/// <returns></returns>
		public static DataTable GetReagents(string _sDivisionCode, DateTime ExpirationDateBefore, DateTime ExpirationDateAfter, DateTime ReceivedDateBefore, DateTime ReceivedDateAfter)
		{
			return DAL.Reagent.GetReagents(_sDivisionCode, ExpirationDateBefore, ExpirationDateAfter, ReceivedDateBefore, ReceivedDateAfter);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6653"> 
		///		<ExpectedInput>Valid guid</ExpectedInput>
		///		<ExpectedOutput>valid row full of values!!</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6654"> 
		///		<ExpectedInput>Invalid guid</ExpectedInput>
		///		<ExpectedOutput>new row with empty values</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrns a reagent data row from it's guid 
		/// </summary>
		/// <param name="reagentGuid"></param>
		/// <returns></returns>
		public static DataRow GetReagent(Guid reagentGuid)
		{
			DataTable dt = DAL.Reagent.GetReagentByGuid(reagentGuid);
			if (dt.Rows.Count>0)
			{
				return dt.Rows[0];
			}
			return dt.NewRow();
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1076"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="677"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Initializes Reagents/Manufacturers table
		/// </summary>
		/// <returns></returns>
		public static DataTable InitializeReagentsAndManufacturersTable()
		{
			DataTable dtReagents = new DataTable("Reagents");
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentGuid, typeof(System.Guid));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentTypeId);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentName);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentManufacturerName);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReceiptDate, typeof(System.DateTime));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReceiptTechId);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.InvoiceNumber);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.LotNumber);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.LotExpirationDate, typeof(System.DateTime));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.NumberVialsReceived, typeof(int));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.NumberVialsAvailable, typeof(int));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.Comments);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.InspectionIndicator, typeof(bool));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.DivisionCode);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.RecordStatusCode);
			dtReagents.Columns.Add(Common.VbecsTables.ReagentType.ReagentTypeName);

			return dtReagents;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1077"> 
		///		<ExpectedInput>Valid DataTable and division code</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2003"> 
		///		<ExpectedInput>Empty data table</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inserts new reagent information 
		/// </summary>
		/// <param name="dtReagentsAndManufacturers"></param>
		/// <param name="dtReagentTypes"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public static bool InsertNewReagent(DataTable dtReagentsAndManufacturers, DataTable dtReagentTypes, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable dtReagents;
			DataRow drReagent;
			bool bResult;
			

			if (dtReagentsAndManufacturers.Rows.Count == 0)
				return false;

			dtReagents = new DataTable("Reagent");
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentGuid, typeof(System.Guid));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentTypeId);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentName);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReagentManufacturerName);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReceiptDate, typeof(System.DateTime));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.ReceiptTechId);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.InvoiceNumber);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.LotNumber);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.LotExpirationDate, typeof(System.DateTime));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.NumberVialsReceived, typeof(int));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.NumberVialsAvailable, typeof(int));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.Comments);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.InspectionIndicator, typeof(bool));
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.DivisionCode);
			dtReagents.Columns.Add(Common.VbecsTables.Reagent.RecordStatusCode);
			System.Guid [] reagentGuids = new Guid[dtReagentsAndManufacturers.Rows.Count];
			int nCtr = 0;
			foreach(DataRow drReagentAndManufacturer in dtReagentsAndManufacturers.Rows)
			{

				drReagent = dtReagents.NewRow();
				drReagent[Common.VbecsTables.Reagent.ReagentGuid] = drReagentAndManufacturer[Common.VbecsTables.Reagent.ReagentGuid];

				reagentGuids[nCtr] = (System.Guid) drReagentAndManufacturer[Common.VbecsTables.Reagent.ReagentGuid];

				drReagent[Common.VbecsTables.Reagent.ReagentTypeId] = drReagentAndManufacturer[Common.VbecsTables.Reagent.ReagentTypeId];
				drReagent[Common.VbecsTables.Reagent.ReagentName] = drReagentAndManufacturer[Common.VbecsTables.Reagent.ReagentName];
				drReagent[Common.VbecsTables.Reagent.ReagentManufacturerName] = drReagentAndManufacturer[Common.VbecsTables.Reagent.ReagentManufacturerName];
				drReagent[Common.VbecsTables.Reagent.ReceiptDate] = drReagentAndManufacturer[Common.VbecsTables.Reagent.ReceiptDate];
				drReagent[Common.VbecsTables.Reagent.ReceiptTechId] = drReagentAndManufacturer[Common.VbecsTables.Reagent.ReceiptTechId];
				drReagent[Common.VbecsTables.Reagent.InvoiceNumber] = drReagentAndManufacturer[Common.VbecsTables.Reagent.InvoiceNumber];
				drReagent[Common.VbecsTables.Reagent.LotNumber] = drReagentAndManufacturer[Common.VbecsTables.Reagent.LotNumber];
				drReagent[Common.VbecsTables.Reagent.LotExpirationDate] = drReagentAndManufacturer[Common.VbecsTables.Reagent.LotExpirationDate];
				drReagent[Common.VbecsTables.Reagent.NumberVialsReceived] = drReagentAndManufacturer[Common.VbecsTables.Reagent.NumberVialsReceived];
				drReagent[Common.VbecsTables.Reagent.NumberVialsAvailable] = drReagentAndManufacturer[Common.VbecsTables.Reagent.NumberVialsAvailable];
				drReagent[Common.VbecsTables.Reagent.Comments] = drReagentAndManufacturer[Common.VbecsTables.Reagent.Comments];
				drReagent[Common.VbecsTables.Reagent.InspectionIndicator] = drReagentAndManufacturer[Common.VbecsTables.Reagent.InspectionIndicator];
				drReagent[Common.VbecsTables.Reagent.DivisionCode] = drReagentAndManufacturer[Common.VbecsTables.Reagent.DivisionCode];
				drReagent[Common.VbecsTables.Reagent.RecordStatusCode] = drReagentAndManufacturer[Common.VbecsTables.Reagent.RecordStatusCode];

				

				dtReagents.Rows.Add(drReagent);
				nCtr = nCtr + 1;
			}

			//CR3508 - removed try/catch - no expected errors for "insert"
			System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.LoginReagent, reagentGuids, Common.WorkloadTransactionType.Miscellanious, false);
 
			bResult = DAL.Reagent.InsertReagents(dtReagents, dtWorkloadEvents, lastUpdateFunctionId);

			return bResult;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1078"> 
		///		<ExpectedInput>Valid ReagentChange object</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2004"> 
		///		<ExpectedInput>Invalid ReagentChange object</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates existing reagent information  
		/// </summary>
		/// <param name="reagentChangeInsert"></param>
		/// <param name="dtReagentList"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public bool UpdateReagent(ReagentChange reagentChangeInsert, DataTable dtReagentList, 
									Common.UpdateFunction lastUpdateFunctionId)
		{
			bool bResult;
			//Removed - duplicate of public property
			
			

			DataTable dtUpdateReagent = new DataTable();
			dtUpdateReagent.Columns.Add(Common.VbecsTables.Reagent.ReagentGuid, typeof(System.Guid));
			dtUpdateReagent.Columns.Add(Common.VbecsTables.Reagent.InspectionIndicator, typeof(bool));
			dtUpdateReagent.Columns.Add(Common.VbecsTables.Reagent.NumberVialsAvailable, typeof(short));
			dtUpdateReagent.Columns.Add(Common.VbecsTables.Reagent.Comments, typeof(string));
			dtUpdateReagent.Columns.Add(Common.VbecsTables.Reagent.RowVersion, typeof(byte[]));

			DataRow drUpdateReagent = dtUpdateReagent.NewRow();
			drUpdateReagent[Common.VbecsTables.Reagent.ReagentGuid] = this.ReagentGuid;

			if (this.Inspection == Common.Inspection.Satisfactory)
			{
				drUpdateReagent[Common.VbecsTables.Reagent.InspectionIndicator] = true;
			}
			else
			{
				drUpdateReagent[Common.VbecsTables.Reagent.InspectionIndicator] = false;
			}

			drUpdateReagent[Common.VbecsTables.Reagent.NumberVialsAvailable] = this.NumberVialsAvailable;
			drUpdateReagent[Common.VbecsTables.Reagent.Comments] = this.Comments;
			drUpdateReagent[Common.VbecsTables.Reagent.RowVersion] = this.RowVersion;

			DataTable dtInsertReagentChange = ReagentChange.InitializeReagentChangeTable();
			DataRow drInsertReagentChange = dtInsertReagentChange.NewRow();

			

			dtInsertReagentChange.Rows.Add(reagentChangeInsert.LoadDataRowFromThis(drInsertReagentChange));
			dtUpdateReagent.Rows.Add(drUpdateReagent);

			int nCtr = 0;
			System.Guid [] reagentGuids = new Guid[dtUpdateReagent.Rows.Count];

			foreach(System.Data.DataRow dtRow in dtUpdateReagent.Rows)
			{
				reagentGuids[nCtr] = (System.Guid) dtRow[Common.VbecsTables.Reagent.ReagentGuid];
				nCtr = nCtr + 1;
			}
			System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.UpdateReagentInventory, reagentGuids, Common.WorkloadTransactionType.Miscellanious, false);

			bResult = DAL.Reagent.UpdateReagent(dtUpdateReagent, dtInsertReagentChange, dtWorkloadEvents, lastUpdateFunctionId);

			return bResult;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1079"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="675"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a DataTable of reagent manufacturers
		/// </summary>
		/// <returns></returns>
		public static DataTable GetReagentManufacturers()
		{
			return DAL.ReagentManufacturer.GetListOfManufacturers();
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1080"> 
		///		<ExpectedInput>Valid division code and manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2005"> 
		///		<ExpectedInput>Invalid division code and invalid manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a DataTable of manufacturer reagent names
		/// </summary>
		/// <param name="manufacturerName"></param>
		/// <returns></returns>
		public static DataTable GetManufacturerReagentNames(string manufacturerName)
		{
			return DAL.Reagent.GetManufacturerReagentNames(manufacturerName);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1081"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2006"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>-1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns an int of reagent type stock level
		/// </summary>
		/// <returns></returns>
		public int GetReagentTypeStockLevel()
		{
			DataTable dtReagentStockLevel = DAL.Reagent.GetReagentTypeStockLevel(this.ReagentType.ReagentTypeId);
			if (dtReagentStockLevel.Rows.Count > 0)
			{
				if (dtReagentStockLevel.Rows[0].IsNull(Common.DatabaseConstants.ArtificialColumnNames.NumberVialsInStock) == false)
				{
					return System.Convert.ToInt32(dtReagentStockLevel.Rows[0][Common.DatabaseConstants.ArtificialColumnNames.NumberVialsInStock]);
				}
				else
					return -1;
			}
			else
				return -1;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1086"> 
		///		<ExpectedInput>Valid division code and reagent type ID</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2007"> 
		///		<ExpectedInput>Invalid division code and invalid reagent type id</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///Returns a DataTable  of reagents by type
		///Implements BR_41.04
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="reagentTypeId"></param>
		/// <returns></returns>
		public static DataTable GetReagentsByType(string divisionCode, int reagentTypeId)
		{
			DataTable dtReagents = DAL.Reagent.GetReagentsByType(divisionCode, reagentTypeId);
			return dtReagents;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1289"> 
		///		<ExpectedInput>Rack identifier</ExpectedInput>
		///		<ExpectedOutput>The data table containing the list of reagent lot numbers for retested rack</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2008"> 
		///		<ExpectedInput>Invalid rack identifier</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of reagent lot numbers for retested rack
		/// </summary>
		/// <param name="RackGuid"></param>
		/// <returns></returns>
		public static DataTable GetReagentLotsForRetest(Guid RackGuid)
		{
			return DAL.Reagent.GetReagentLotsForRetest(RackGuid);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1296"> 
		///		<ExpectedInput>Rack name</ExpectedInput>
		///		<ExpectedOutput>The data table containing the list of reagent lot numbers from a previous test for a tested rack</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2009"> 
		///		<ExpectedInput>Invalid rack name</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of reagent lot numbers from a previous test for a tested rack
		/// </summary>
		/// <param name="rackName"></param>
		/// <returns></returns>
		public static DataTable GetReagentLotsFromPreviousTest(string rackName)
		{
			return DAL.Reagent.GetReagentLotsFromPreviousTest(rackName);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6655"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6656"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the reagent lot number used for a tested rack
		/// </summary>
		/// <param name="RackGuid">Rack GUID</param>
		/// <param name="reagentTypeID">Reagent Type ID</param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns>Data table</returns>
		public static DataTable GetReagentFromRack(Guid RackGuid,int reagentTypeID, string divisionCode)
		{
			return DAL.Reagent.GetReagentFromRack(RackGuid, reagentTypeID, divisionCode);
		}


		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6657"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6658"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of reagents for the invoice
		/// </summary>
		/// <param name="invoiceNumber"></param>
		/// <returns></returns>
		public static DataTable GetReagentsForInvoice(string invoiceNumber)
		{
			return DAL.Reagent.GetReagentsForInvoice(invoiceNumber);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1256"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="2872"> 
		///		<ExpectedInput>Bad DataRow object (with missing columns)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			dtRow[Common.VbecsTables.Reagent.ReagentGuid] = this.ReagentGuid;
			dtRow[Common.VbecsTables.Reagent.ReagentTypeId] = this.ReagentType.ReagentTypeId;
			dtRow[Common.VbecsTables.Reagent.ReagentName] = this.ReagentName;
			dtRow[Common.VbecsTables.Reagent.ReceiptDate] = this.ReceiptDate;
			dtRow[Common.VbecsTables.Reagent.InvoiceNumber] = this.InvoiceNumber;
			dtRow[Common.VbecsTables.Reagent.InspectionIndicator] = this.Inspection;
			dtRow[Common.VbecsTables.Reagent.LotNumber] = this.LotNumber;
			dtRow[Common.VbecsTables.Reagent.LotExpirationDate] = this.LotExpirationDate;
			dtRow[Common.VbecsTables.Reagent.NumberVialsReceived] = this.NumberVialsReceived;
			dtRow[Common.VbecsTables.Reagent.NumberVialsAvailable] = this.NumberVialsAvailable;
			dtRow[Common.VbecsTables.Reagent.Comments] = this.Comments;
			dtRow[Common.VbecsTables.Reagent.RowVersion] = this.RowVersion;

			return dtRow;
		}


		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if(this.VerifyMinColumnsInDataTable(REAGENTCOLUMNS.Split(','), dtRow.Table))
			{
				this.ReagentGuid = dtRow.IsNull(Common.VbecsTables.Reagent.ReagentGuid) ? System.Guid.Empty : (System.Guid)dtRow[Common.VbecsTables.Reagent.ReagentGuid];
				this.ReagentName = dtRow.IsNull(Common.VbecsTables.Reagent.ReagentName) ? string.Empty : dtRow[Common.VbecsTables.Reagent.ReagentName].ToString();
				this.LotNumber = dtRow.IsNull(Common.VbecsTables.Reagent.LotNumber) ? string.Empty : dtRow[Common.VbecsTables.Reagent.LotNumber].ToString();
				this.LotExpirationDate = dtRow.IsNull(Common.VbecsTables.Reagent.LotExpirationDate) ? DateTime.MinValue : (DateTime)dtRow[Common.VbecsTables.Reagent.LotExpirationDate];
				this.ReagentManufacturerName = dtRow.IsNull(Common.VbecsTables.Reagent.ReagentManufacturerName) ? string.Empty : dtRow[Common.VbecsTables.Reagent.ReagentManufacturerName].ToString();
				if (dtRow.IsNull(Common.VbecsTables.Reagent.InspectionIndicator))
				{
					this.Inspection = Common.Inspection.UnSatisfactory;
				}
				else
				{
					if ((bool)dtRow[Common.VbecsTables.Reagent.InspectionIndicator] == true)
					{
						this.Inspection = Common.Inspection.Satisfactory;
					}
					else
					{
						this.Inspection = Common.Inspection.UnSatisfactory;
					}
				}
			}
			else
			{
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString);	
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.Reagent.ReagentTypeId))
			{
				this.ReagentType.ReagentTypeId = dtRow.IsNull(Common.VbecsTables.Reagent.ReagentTypeId) ? 0 : (int)dtRow[Common.VbecsTables.Reagent.ReagentTypeId];
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.ReagentType.ReagentTypeName))
			{
				this.ReagentType.ReagentTypeName = dtRow.IsNull(Common.VbecsTables.ReagentType.ReagentTypeName) ? "" : dtRow[Common.VbecsTables.ReagentType.ReagentTypeName].ToString();
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.Reagent.ReceiptDate))
			{
				this.ReceiptDate = dtRow.IsNull(Common.VbecsTables.Reagent.ReceiptDate) ? DateTime.MinValue : (DateTime)dtRow[Common.VbecsTables.Reagent.ReceiptDate];
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.Reagent.InvoiceNumber))
			{
				this.InvoiceNumber = dtRow.IsNull(Common.VbecsTables.Reagent.InvoiceNumber) ? string.Empty : dtRow[Common.VbecsTables.Reagent.InvoiceNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.Reagent.NumberVialsReceived))
			{
				this.NumberVialsReceived = dtRow.IsNull(Common.VbecsTables.Reagent.NumberVialsReceived) ? short.MinValue : (short)dtRow[Common.VbecsTables.Reagent.NumberVialsReceived];
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.Reagent.NumberVialsAvailable))
			{
				this.NumberVialsAvailable = dtRow.IsNull(Common.VbecsTables.Reagent.NumberVialsAvailable) ? short.MinValue : (short)dtRow[Common.VbecsTables.Reagent.NumberVialsAvailable];
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.Reagent.Comments))
			{
				this.Comments = dtRow.IsNull(Common.VbecsTables.Reagent.Comments) ? string.Empty : dtRow[Common.VbecsTables.Reagent.Comments].ToString();	
			}
			if (dtRow.Table.Columns.Contains(Common.VbecsTables.Reagent.RowVersion))
			{
				this.RowVersion = dtRow.IsNull(Common.VbecsTables.Reagent.RowVersion) ? new byte[0] : (byte[])dtRow[Common.VbecsTables.Reagent.RowVersion];
			}
		}



		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1258"> 
		///		<ExpectedInput>string array of column names, Valid data table</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="761"> 
		///		<ExpectedInput>invalid datatable</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks the columns in the data table against a string array of column names
		/// </summary>
		public bool VerifyMinColumnsInDataTable(string [] colNames, System.Data.DataTable dtTable)
		{
			return Common.Utility.VerifyMinColumnsInDataTable(REAGENTCOLUMNS.Split(','), dtTable);
		}	

	}
}



